// Scout Tool Sample Program 4: Light Bloodhound
// Copyright The LEGO Company - November 18, 1999

// Description: Uses the light sensor, variables, and seek subroutine to sample
// a bright light source and find it again.  Best used on Bug left-right drive 
// subassembly.

// Requires "scoutdef.h" for shortcut commands like on, rwd, fwd.
// See Scout SDK documentation for formal assembler commands.

// Special Comments:

// Use a dark room and a bright flashlight
// Place light on level stand about 4 cm high.
// Press run.
// Place Scout within lightbeam at 30 cm range.
// Press Touch 1 to sample.
// Move Scout to another location and 
// Press Touch 2 to start search.
// Scout should find light and stop within
// about 30 cm before hitting flash light.
//--------------------------------------------------------------------

// Program:

#include "ScoutDef.h"

dels                          //  deletes all subroutines from memory
delt                           // deletes all tasks from memory

// local variables defined for use in program:
   
#define Arg1 10
#define Arg2 11
#define Arg3 12
#define Arg4 13
#define TmpLight 14
#define Pause 15
#define Light 0

// Geiger counter:  uses task 1 to run the geiger counter
// subroutine in parallel with the main program
 
task 1                            // start of task 1        
     setv Arg1,2,200       // uses a local variable to pass a value to the Geiger Counter Subroutine
     calls 13                    // calls subroutine #13 (the Geiger Coutner)
endt                              // end of task 1

// Main Program:

// Step 1: Start of Program

task 0                           // start of task 0
     plays 6                     // plays System Sound #6 

// Step 2: User presses touch sensor 1 to sample light level from a single light source.

     mone SRC_CON, EVENT_T1PR, endWait1
                         // " mone" starts a monitor for touch 1 press event
                         // jumps to "endWait1" label when the event is triggered

Wait1:               // start of Wait1 loop
     jmp Wait1     // jumps to label "Wait1" to repeat the loop   
endWait1:         // jump point for when event monitor breaks out of the "Wait1" loop 

     plays 4          // plays System Sound #4 as a signal touch 1 has been pressed
     
     setv Light, SRC_SENRAW, SEN_LIGHT
                          // reads the light level and records it as a local variable called "Light"

     start 1           // starts Task 1 running, which is the Light Geiger subroutine
                          // which plays a high pitched sound which increase with the light level

// Step 3: User places rover in different location away from light source
// and presses touch sensor 2 to active Scout in its search routine.

     mone	SRC_CON, EVENT_T2PR, endWait2
                         // makes program wait for press event on touch 2 (see example above)
Wait2:
     jmp Wait2
endWait2:

// Step 4:  Scout does an initial spin left and right

     fwd 2                            // sets B to forward
     rwd 1                            // sets A to backward 
     on 3                              // turns A and B on
     wait SRC_CON, 200    //  waits 2 seconds
     alt 3                              // changes the direction of the motors on AB
     wait SRC_CON, 300     // waits 3 seconds
     off 3                              // turns A and B off

// Step 4:  Light source search routine.  Scout uses seek light subroutine
// to find the light sources and travels towards it and a narrowing search.

    setv Arg2, SRC_CON, 8    // set a local variable to pass a value to the seek subroutine

repeat:
     setv Arg1, SRC_CON, 1                            // sets local varibale to set seek subroutine to seek light
     calls SUB_SEEK  			              // calls Seek subroutine
  	setv 	TmpLight, SRC_SENRAW,SEN_LIGHT        // records current light reading as "TmpLight"
	subv 	TmpLight, SRC_VAR, Light              // subtracts the initial light reading level from the current reading
                                                      // Note: the raw light reading is from 0 to 1020, with brighter values being lower.
                                                      // The closer the current light reading gets to the original bright light,
                                                      // the smaller the difference will get.
                subv 	TmpLight, SRC_CON,50          // subtracts another 50 units from TmpLight 
	decvjn 	TmpLight, until                       // jumps to "until" label if the TmpLight falls below zero.
                                                      // (This will occur when the Scout gets close to the original light source.
                                                      // The subtraction of 50 is used as a buffer in case the difference 
                                                      // gets close but not below zero.)
                sumv        TmpLight, SRC_CON,50      // adds 50 back to TmpLight
	setv 	Pause, SRC_VAR, TmpLight              // sets the variable "Pause" to equal TmpLight
	divv 	Pause, SRC_CON, 2                     // divides Pause by 2
	fwd 3                                         // sets direction of A and B forward
	on  3                                         // turns A and B on
	wait 	SRC_VAR, Pause			      // waits for an amount time based on the value of "Pause" (defined in 100ths of a second.)
	                                              // (Using the Pause variable decreases the amount of time between Seek Light routines as
                                                      // as the Scout gets closer to the light source, thus narrowing the search.)
        jmp 	repeat                                // returns to the "repeat" label to restart the loop
until:                                                // jump to point for when the light level gets close to the initial value
 
// Step 5:  Scout does a little dance once it has found the light.

             stop 	1                             // stops Task 1 to end the Geiger Counter subroutine
 	plays 6                                       // plays System Sound #6 as a signal
  	light 1                                       // turns the Scout's red light on
                wait 	SRC_CON,50                    // waits .5 seconds
  	setv 	Arg1, SRC_CON, 3                      // sets two variables for parameters to pass to the Movements subroutine
  	setv 	Arg2, SRC_CON, 2
  	calls 	SUB_MOVEMENTS                         // calls the Movements subroutine
                light 0                               // turns the Scout's light off
                fwd 1                                 // sets the direction of A forward
                on 1                                  // turns A on
                wait 	SRC_CON,50                    // waits .5 seconds
                off 1                                 // turns A off
                plays 6                               // plays System Sound #6 to signal the end of the program
endt                                                  // end of Task 0
    