// Scout Tool Sample Program 5: Access Control Demonstration
// Copyright The LEGO Company - November 18, 1999

// Description: Demonstrates how to use priority commands with 
// access control to manange what part of the program is in
// control of the motor and sound resources.

// In this case, the main task is divided into 4 stages with increasing priority,
// (note: in LEGO assembler code a priority of 0 has the highest priority and a priority 
// of 7 has the lowest.)  Each stage is 3 seconds long and begins with a tone.
// In the first stage, a press event on touch 1, a press event on touch 2, or a
// see bright event will all override the main task.  In stage 2, only the two touch
// events can override the main task.  In stage 3, only a touch 1 event can 
// override the main task.  In stage 4, no event can override the main task.
// When an event overrides the main task, the main task is suspended until the event 
// task completes its sequence.  Control returns to the main task at the beginning
// of the access control region (or stage in our example) that was interrupted.

// The Scout system supports a wide variety of scenarios based on how the access
// control regions, priorities, and resume labels are defined.  
// This is just one example.

// Requires "scoutdef.h" for shortcut commands like on, rwd, fwd.
// See Scout SDK documentation for formal assembler commands.

// Program:

#include "ScoutDef.h"

dels	// deletes all prior subroutines in memory
delt	// deletes all prior tasks in memory

task 0                       // Start of Task 0 (the Main Task)
// ***** init *****
   lscal
   start 1                   // starts task 1
   start 2                   // starts task 2
   start 3                   // starts task 3

// ***** main task *****

// Access Control Region 1:
// Does not have priority over light event, touch 2, or touch 1.

resume1:                      // resume label for where the program should restart if it is interrupted
   setp 6                       // priority level for this ACR
   monal RES_ABS, resume1     // registers a claim for control over resources
                                                  // for sound and outputs A and B 
        playt 700,100         // plays a tone to indicate the stage
        fwd 3
        on 3
        wait SRC_CON,300
        off 3
   monax                       // releases control over resources


// Access Control Region 2:  same as Region 1 except the
// priority increases to 4 and plays a higher tone.
// Has priority over light event, but not touch 2 nor touch 1.

resume2:   
   setp 4
   monal RES_ABS, resume2
        playt 800,100
        fwd 3
        on 3
        wait SRC_CON,300
        off 3
   monax  


// Access Control Region 3:  same as Region 1 except the
//  priority increases to 2 and plays a higher tone.
// Has priority over light event, and touch 2, but not touch 1.

resume3:   
   setp 2
   monal RES_ABS, resume3
        playt 900,100 
        fwd 3
        on 3
        wait SRC_CON,300
        off 3
   monax  


// Access Control Region 4:  same as Region 1 except the
// priority increases to 0 and plays a higher tone/
// Has priority over the other events.

resume4:   
   setp 0
   monal RES_ABS, resume4
        playt 1000,100 
        fwd 3
        on 3
        wait SRC_CON,300
        off 3
stop 3                                     // discontinues other tasks
stop 2
stop 1
   monax  

stop 0
endt                                        // End of Task 0


// Touch 1 Event Watcher:  Interrupts any other task with equal or
//  lesser priority, including itself.
// Priority set to 3, does avoid movement with right turn

task 1
start1:
     mone SRC_CON, EVENT_T1PR, end01       
                         // monitors for a touch event and jumps out of loop upon press 1
start01:
   jmp start01
end01:
     mone SRC_CON, EVENT_T1PR, end01       
                             // immediately restarts monitor once event has been detected
     setp 1                                   // sets priority for access control region
     monal RES_ABS, resume1              
                                                  // defining beginning of access control region
          plays 2                             // sound and motion sequence for avoid
          rwd 3
          on 3
          wait SRC_CON,200
          fwd 1
          wait SRC_CON,50
          fwd  2
          off 3
          plays 6
     monax
resume1:            // resume point for when access control region is interrupted
                          // in this case, causes task to resume control at the end
                          // so it will not restart the full sequence after being interrupted
   jmp start1
endt


// Touch 2 Event Watcher: same basic structure as Touch 1
// Priority set to 3, does avoid movement with left turn

task 2
start2:
     mone SRC_CON, EVENT_T2PR, end02       //
start02:
   jmp start02
end02:
     mone SRC_CON, EVENT_T1PR, end02
     setp 3
     monal RES_ABS, resume2
          plays 3
          rwd 3
          on 3
          wait SRC_CON,200
          fwd 2
          wait SRC_CON,50
          off 3
          plays 6
     monax
resume2:
   jmp start2
endt

// Light Event Watcher: same basic structure as Touch 1
// Priority set to 5, does back up movement with the Bug Dance

task 3
start3:
     mone SRC_CON, EVENT_LGT, end03
start03: 
   jmp start03
end03:
     mone SRC_CON, EVENT_LGT, end03
     setp 5
     monal RES_ABS, resume3
          plays 27
          rwd 3
          on 3
          wait SRC_CON,50
          setv 10,SRC_CON,0
          setv 11,SRC_CON,100
          calls 6
          fwd 3
          off 3
          plays 6
     monax
resume3:
   jmp start3
endt



